import { Suspense } from "react"
import { notFound } from "next/navigation"
import { Header } from "@/components/header"
import { TVDetails } from "@/components/tv-details"
import { TVTrailer } from "@/components/tv-trailer"
import { WhereToWatchTV } from "@/components/where-to-watch-tv"
import { SimilarTVShows } from "@/components/similar-tv-shows"
import { AdultContentWarning } from "@/components/adult-content-warning"
import { LoginRequiredOverlay } from "@/components/login-required-overlay"
import { Skeleton } from "@/components/ui/skeleton"
import { createServerClient } from "@/lib/supabase"

async function getTVShow(id: string) {
  try {
    const res = await fetch(
      `https://api.themoviedb.org/3/tv/${id}?api_key=${process.env.TMDB_API_KEY}&append_to_response=videos,credits,watch/providers`,
      { next: { revalidate: 3600 } },
    )

    if (!res.ok) {
      console.error(`TMDB TV show fetch failed (${res.status}) for id ${id}`)
      return null
    }

    return res.json()
  } catch (err) {
    console.error("TMDB TV show fetch error:", err)
    return null
  }
}

async function checkUserAuth() {
  try {
    const supabase = createServerClient()
    const {
      data: { session },
    } = await supabase.auth.getSession()
    return !!session?.user
  } catch (error) {
    return false
  }
}

export default async function TVShowPage({
  params,
}: {
  params: { id: string }
}) {
  const [tvShow, isLoggedIn] = await Promise.all([getTVShow(params.id), checkUserAuth()])

  if (!tvShow) {
    notFound()
  }

  // Check if this is adult content and user is not logged in
  if (tvShow.adult && !isLoggedIn) {
    return (
      <div className="min-h-screen bg-background">
        <Header />
        <div className="container mx-auto px-4 py-8">
          <LoginRequiredOverlay
            title={tvShow.name}
            type="tv"
            description="This TV show contains mature content and requires user authentication to view."
          />
        </div>
      </div>
    )
  }

  const tvContent = (
    <>
      <div className="container mx-auto px-4 py-8">
        <TVDetails tvShow={tvShow} />

        <div className="mt-12 grid grid-cols-1 lg:grid-cols-3 gap-8">
          <div className="lg:col-span-2 space-y-8">
            <Suspense fallback={<Skeleton className="aspect-video w-full" />}>
              <TVTrailer videos={tvShow.videos?.results || []} />
            </Suspense>

            <div>
              <h2 className="text-2xl font-bold mb-6">Similar TV Shows</h2>
              <Suspense fallback={<div>Loading similar TV shows...</div>}>
                <SimilarTVShows tvShowId={params.id} />
              </Suspense>
            </div>
          </div>

          <div>
            <WhereToWatchTV providers={tvShow["watch/providers"]?.results || {}} />
          </div>
        </div>
      </div>
    </>
  )

  return (
    <div className="min-h-screen bg-background">
      <Header />

      {tvShow.adult && isLoggedIn ? (
        <AdultContentWarning isAdult={tvShow.adult} title={tvShow.name} type="tv">
          {tvContent}
        </AdultContentWarning>
      ) : (
        tvContent
      )}
    </div>
  )
}

export async function generateMetadata({
  params,
}: {
  params: { id: string }
}) {
  const tvShow = await getTVShow(params.id)

  if (!tvShow) {
    return {
      title: "TV Show Not Found",
    }
  }

  return {
    title: `${tvShow.name} - MoviesWeb`,
    description: tvShow.overview,
    openGraph: {
      title: tvShow.name,
      description: tvShow.overview,
      images: tvShow.poster_path ? [`https://image.tmdb.org/t/p/w500${tvShow.poster_path}`] : [],
    },
  }
}
